/*
 *	PostScript driver initialization functions
 *
 *	Copyright 1998 Huw D M Davies
 *	Copyright 2001 Marcus Meissner
 *
 */

#include "config.h"

#include <string.h>
#include <unistd.h>

#include "gdi.h"
#include "psdrv.h"
#include "debugtools.h"
#include "winreg.h"
#include "winspool.h"
#include "winerror.h"

#ifdef HAVE_CUPS
# include <cups/cups.h>
#endif

DEFAULT_DEBUG_CHANNEL(psdrv);

static PSDRV_DEVMODEA DefaultDevmode = 
{
  { /* dmPublic */
/* dmDeviceName */	"Wine PostScript Driver",
/* dmSpecVersion */	0x30a,
/* dmDriverVersion */	0x001,
/* dmSize */		sizeof(DEVMODEA),
/* dmDriverExtra */	0,
/* dmFields */		DM_ORIENTATION | DM_PAPERSIZE | DM_SCALE | 
			DM_COPIES | DM_DEFAULTSOURCE | DM_COLOR | 
			DM_DUPLEX | DM_YRESOLUTION | DM_TTOPTION,
   { /* u1 */
     { /* s1 */
/* dmOrientation */	DMORIENT_PORTRAIT,
/* dmPaperSize */	DMPAPER_A4,
/* dmPaperLength */	2969,
/* dmPaperWidth */      2101
     }
   },
/* dmScale */		100, /* ?? */
/* dmCopies */		1,
/* dmDefaultSource */	DMBIN_AUTO,
/* dmPrintQuality */	0,
/* dmColor */		DMCOLOR_COLOR,
/* dmDuplex */		0,
/* dmYResolution */	0,
/* dmTTOption */	DMTT_SUBDEV,
/* dmCollate */		0,
/* dmFormName */	"",
/* dmUnusedPadding */   0,
/* dmBitsPerPel */	0,
/* dmPelsWidth */	0,
/* dmPelsHeight */	0,
/* dmDisplayFlags */	0,
/* dmDisplayFrequency */ 0,
/* dmICMMethod */       0,
/* dmICMIntent */       0,
/* dmMediaType */       0,
/* dmDitherType */      0,
/* dmReserved1 */       0,
/* dmReserved2 */       0,
/* dmPanningWidth */    0,
/* dmPanningHeight */   0
  },
  { /* dmDocPrivate */
    /* dummy */ 0
  },
  { /* dmDrvPrivate */
    /* numInstalledOptions */ 0 
  }
};

HANDLE PSDRV_Heap = 0;

static HANDLE PSDRV_DefaultFont = 0;
static LOGFONTA DefaultLogFont = {
    100, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, ANSI_CHARSET, 0, 0,
    DEFAULT_QUALITY, FIXED_PITCH | FF_MODERN, ""
};

/*********************************************************************
 *	     PSDRV_Init
 *
 * Initializes font metrics and registers driver. Called from GDI_Init()
 *
 */
BOOL WINAPI PSDRV_Init( HINSTANCE hinst, DWORD reason, LPVOID reserved )
{
    TRACE("(0x%4x, 0x%08lx, %p)\n", hinst, reason, reserved);
   
    switch(reason) {

	case DLL_PROCESS_ATTACH:

	    PSDRV_Heap = HeapCreate(0, 0x10000, 0);
	    if (PSDRV_Heap == (HANDLE)NULL)
		return FALSE;

	    if (PSDRV_GetFontMetrics() == FALSE) {
		HeapDestroy(PSDRV_Heap);
		return FALSE;
	    }

	    PSDRV_DefaultFont = CreateFontIndirectA(&DefaultLogFont);
	    if (PSDRV_DefaultFont == (HANDLE)NULL) {
		HeapDestroy(PSDRV_Heap);
		return FALSE;
	    }
            break;

	case DLL_PROCESS_DETACH:

	    DeleteObject( PSDRV_DefaultFont );
	    HeapDestroy( PSDRV_Heap );
            break;
    }
 
    return TRUE;
}


/**********************************************************************
 *	     PSDRV_CreateDC
 */
BOOL PSDRV_CreateDC( DC *dc, LPCSTR driver, LPCSTR device,
                     LPCSTR output, const DEVMODEA* initData )
{
    PSDRV_PDEVICE *physDev;
    PRINTERINFO *pi;
    PAGESIZE *page;
    INT width = 0, height = 0;

    /* If no device name was specified, retrieve the device name
     * from the DEVMODE structure from the DC's physDev.
     * (See CreateCompatibleDC) */
    if ( !device && dc->physDev )
    {
        physDev = (PSDRV_PDEVICE *)dc->physDev;
        device = physDev->Devmode->dmPublic.dmDeviceName;
    }
    pi = PSDRV_FindPrinterInfo(device);
        
    TRACE("(%s %s %s %p)\n", driver, device, output, initData);

    if(!pi) return FALSE;

    if(!pi->Fonts) {
        MESSAGE("To use WINEPS you need to install some AFM files.\n");
	return FALSE;
    }

    physDev = (PSDRV_PDEVICE *)HeapAlloc( PSDRV_Heap, HEAP_ZERO_MEMORY,
					             sizeof(*physDev) );
    if (!physDev) return FALSE;
    dc->physDev = physDev;

    physDev->pi = pi;

    physDev->Devmode = (PSDRV_DEVMODEA *)HeapAlloc( PSDRV_Heap, 0,
						     sizeof(PSDRV_DEVMODEA) );
    if(!physDev->Devmode) {
        HeapFree( PSDRV_Heap, 0, physDev );
	return FALSE;
    }
    
    memcpy( physDev->Devmode, pi->Devmode, sizeof(PSDRV_DEVMODEA) );

    if(initData) {
        PSDRV_MergeDevmodes(physDev->Devmode, (PSDRV_DEVMODEA *)initData, pi);
    }

    physDev->logPixelsX = physDev->pi->ppd->DefaultResolution;
    physDev->logPixelsY = physDev->pi->ppd->DefaultResolution;

    for(page = pi->ppd->PageSizes; page; page = page->next) {
        if(page->WinPage == physDev->Devmode->dmPublic.u1.s1.dmPaperSize)
	    break;
    }

    if(!page) {
        FIXME("Can't find page\n");
	physDev->PageSize.left = 0;
	physDev->PageSize.right = 0;
	physDev->PageSize.bottom = 0;
        physDev->PageSize.top = 0;
    } else if(page->ImageableArea) {  /* PageSize is in device units */
        physDev->PageSize.left = page->ImageableArea->llx * physDev->logPixelsX / 72;
        physDev->PageSize.right = page->ImageableArea->urx * physDev->logPixelsX / 72;
        physDev->PageSize.bottom = page->ImageableArea->lly * physDev->logPixelsY / 72;
        physDev->PageSize.top = page->ImageableArea->ury * physDev->logPixelsY / 72;
    } else {
        physDev->PageSize.left = physDev->PageSize.bottom = 0;
        physDev->PageSize.right = page->PaperDimension->x * physDev->logPixelsX / 72;
        physDev->PageSize.top = page->PaperDimension->y * physDev->logPixelsY / 72;
    }
    TRACE("PageSize = (%d,%d - %d,%d)\n",physDev->PageSize.left, physDev->PageSize.bottom, physDev->PageSize.right, physDev->PageSize.top);

    /* these are in device units */
    width = physDev->PageSize.right - physDev->PageSize.left;
    height = physDev->PageSize.top - physDev->PageSize.bottom;

    if(physDev->Devmode->dmPublic.u1.s1.dmOrientation == DMORIENT_PORTRAIT) {
        physDev->horzRes = width;
        physDev->vertRes = height;
    } else {
        physDev->horzRes = height;
        physDev->vertRes = width;
    }

    /* these are in mm */
    physDev->horzSize = (physDev->horzRes * 25.4) / physDev->logPixelsX;
    physDev->vertSize = (physDev->vertRes * 25.4) / physDev->logPixelsY;

    TRACE("devcaps: horzSize = %dmm, vertSize = %dmm, "
	  "horzRes = %d, vertRes = %d\n",
	  physDev->horzSize, physDev->vertSize,
	  physDev->horzRes, physDev->vertRes);

    /* etc */

    dc->hVisRgn = CreateRectRgn(0, 0, physDev->horzRes, physDev->vertRes);
    dc->hFont = PSDRV_DefaultFont;

    if (!output) output = "LPT1:";  /* HACK */
    physDev->job.output = HeapAlloc( PSDRV_Heap, 0, strlen(output)+1 );
    strcpy( physDev->job.output, output );
    physDev->job.hJob = 0;
    return TRUE;
}


/**********************************************************************
 *	     PSDRV_DeleteDC
 */
BOOL PSDRV_DeleteDC( DC *dc )
{
    PSDRV_PDEVICE *physDev = (PSDRV_PDEVICE *)dc->physDev;
    
    TRACE("\n");

    HeapFree( PSDRV_Heap, 0, physDev->Devmode );
    HeapFree( PSDRV_Heap, 0, physDev->job.output );
    HeapFree( PSDRV_Heap, 0, physDev );
    dc->physDev = NULL;

    return TRUE;
}


/***********************************************************************
 *           get_phys_page_size
 *
 * Helper function to compute PHYSICALWIDTH and PHYSICALHEIGHT dev caps.
 */
static void get_phys_page_size( const PSDRV_PDEVICE *pdev, POINT *p )
{
    p->x = p->y = 0;

    if ((pdev->Devmode->dmPublic.dmFields & DM_PAPERSIZE) != 0 &&
        pdev->Devmode->dmPublic.u1.s1.dmPaperSize != 0)
    {
        PAGESIZE *page = pdev->pi->ppd->PageSizes;

        while (page != NULL)
        {
            if (page->WinPage == pdev->Devmode->dmPublic.u1.s1.dmPaperSize)
                break;
            page = page->next;
        }

        if (page == NULL)
        {
            ERR("No entry for papersize %u in PPD file for '%s'\n",
                pdev->Devmode->dmPublic.u1.s1.dmPaperSize,
                pdev->pi->FriendlyName);
            return;
        }

        TRACE("Found '%s' for paper size %u\n", page->FullName,
              pdev->Devmode->dmPublic.u1.s1.dmPaperSize);

        p->x = page->PaperDimension->x * pdev->logPixelsX / 72;
        p->y = page->PaperDimension->y * pdev->logPixelsY / 72;

        TRACE("%fx%f PostScript points = %lix%li device units\n",
              page->PaperDimension->x, page->PaperDimension->y,
              p->x, p->y);
    }

    /* These are in tenths of a millimeter */
    if ((pdev->Devmode->dmPublic.dmFields & DM_PAPERWIDTH) != 0 &&
        pdev->Devmode->dmPublic.u1.s1.dmPaperWidth != 0)
    {
        p->x = (pdev->Devmode->dmPublic.u1.s1.dmPaperWidth *
                pdev->logPixelsX) / 254;
        TRACE("dmPaperWidth = %li device units\n", p->x);
    }

    if ((pdev->Devmode->dmPublic.dmFields & DM_PAPERLENGTH) != 0 &&
        pdev->Devmode->dmPublic.u1.s1.dmPaperLength != 0)
    {
        p->y = (pdev->Devmode->dmPublic.u1.s1.dmPaperLength *
                pdev->logPixelsY) / 254;
        TRACE("dmPaperLength = %li device units\n", p->y);
    }

    if (p->x == 0 || p->y == 0)
    {
        ERR("Paper size not properly set for '%s'\n", pdev->pi->FriendlyName);
        return;
    }

    if ((pdev->Devmode->dmPublic.dmFields & DM_ORIENTATION) != 0 &&
        pdev->Devmode->dmPublic.u1.s1.dmOrientation == DMORIENT_LANDSCAPE)
    {
        INT temp = p->y;
        p->y = p->x;
        p->x = temp;
    }
}


/***********************************************************************
 *           GetDeviceCaps    (WINEPS.@)
 */
INT PSDRV_GetDeviceCaps( DC *dc, INT cap )
{
    PSDRV_PDEVICE *physDev = dc->physDev;
    POINT pt;

    switch(cap)
    {
    case DRIVERVERSION:
        return 0;
    case TECHNOLOGY:
        return DT_RASPRINTER;
    case HORZSIZE:
        return physDev->horzSize;
    case VERTSIZE:
        return physDev->vertSize;
    case HORZRES:
        return physDev->horzRes;
    case VERTRES:
        return physDev->vertRes;
    case BITSPIXEL:
        return (physDev->pi->ppd->ColorDevice ? 8 : 1);
    case PLANES:
        return 1;
    case NUMBRUSHES:
        return -1;
    case NUMPENS:
        return 10;
    case NUMMARKERS:
        return 0;
    case NUMFONTS:
        return 39;
    case NUMCOLORS:
        return (physDev->pi->ppd->ColorDevice ? 256 : -1);
    case PDEVICESIZE:
        return sizeof(PSDRV_PDEVICE);
    case CURVECAPS:
        return (CC_CIRCLES | CC_PIE | CC_CHORD | CC_ELLIPSES | CC_WIDE |
                CC_STYLED | CC_WIDESTYLED | CC_INTERIORS | CC_ROUNDRECT);
    case LINECAPS:
        return (LC_POLYLINE | LC_MARKER | LC_POLYMARKER | LC_WIDE |
                LC_STYLED | LC_WIDESTYLED | LC_INTERIORS);
    case POLYGONALCAPS:
        return (PC_POLYGON | PC_RECTANGLE | PC_WINDPOLYGON | PC_SCANLINE |
                PC_WIDE | PC_STYLED | PC_WIDESTYLED | PC_INTERIORS);
    case TEXTCAPS:
        return TC_CR_ANY; /* psdrv 0x59f7 */
    case CLIPCAPS:
        return CP_RECTANGLE;
    case RASTERCAPS:
        return (RC_BITBLT | RC_BITMAP64 | RC_GDI20_OUTPUT | RC_DIBTODEV |
                RC_STRETCHBLT | RC_STRETCHDIB); /* psdrv 0x6e99 */
    /* Are aspect[XY] and logPixels[XY] correct? */
    /* Need to handle different res in x and y => fix ppd */
    case ASPECTX:
    case ASPECTY:
        return physDev->pi->ppd->DefaultResolution;
    case ASPECTXY:
        return (int)hypot( (double)physDev->pi->ppd->DefaultResolution,
                           (double)physDev->pi->ppd->DefaultResolution );
    case LOGPIXELSX:
        return physDev->logPixelsX;
    case LOGPIXELSY:
        return physDev->logPixelsY;
    case SIZEPALETTE:
        return 0;
    case NUMRESERVED:
        return 0;
    case COLORRES:
        return 0;
    case PHYSICALWIDTH:
        get_phys_page_size( physDev, &pt );
        return pt.x;
    case PHYSICALHEIGHT:
        get_phys_page_size( physDev, &pt );
        return pt.y;
    case PHYSICALOFFSETX:
    case PHYSICALOFFSETY:
    case SCALINGFACTORX:
    case SCALINGFACTORY:
    case VREFRESH:
    case DESKTOPVERTRES:
    case DESKTOPHORZRES:
    case BTLALIGNMENT:
        return 0;
    default:
        FIXME("(%04x): unsupported capability %d, will return 0\n", dc->hSelf, cap );
        return 0;
    }
}


/**********************************************************************
 *		PSDRV_FindPrinterInfo
 */
PRINTERINFO *PSDRV_FindPrinterInfo(LPCSTR name) 
{
    static PRINTERINFO *PSDRV_PrinterList;
    DWORD type = REG_BINARY, needed, res, dwPaperSize;
    PRINTERINFO *pi = PSDRV_PrinterList, **last = &PSDRV_PrinterList;
    FONTNAME *font;
    const AFM *afm;
    HANDLE hPrinter;
    const char *ppd = NULL;
    char ppdFileName[256];
    HKEY hkey;

    TRACE("'%s'\n", name);
    
    /*
     *	If this loop completes, last will point to the 'next' element of the
     *	final PRINTERINFO in the list
     */    
    for( ; pi; last = &pi->next, pi = pi->next)
        if(!strcmp(pi->FriendlyName, name))
	    return pi;

    pi = *last = HeapAlloc( PSDRV_Heap, HEAP_ZERO_MEMORY, sizeof(*pi) );
    if (pi == NULL)
    	return NULL;

    if (!(pi->FriendlyName = HeapAlloc( PSDRV_Heap, 0, strlen(name)+1 ))) goto fail;
    strcpy( pi->FriendlyName, name );

    /* Use Get|SetPrinterDataExA instead? */
    
    res = DrvGetPrinterData16((LPSTR)name, (LPSTR)INT_PD_DEFAULT_DEVMODE, &type,
			    NULL, 0, &needed );

    if(res == ERROR_INVALID_PRINTER_NAME || needed != sizeof(DefaultDevmode)) {
        pi->Devmode = HeapAlloc( PSDRV_Heap, 0, sizeof(DefaultDevmode) );
	if (pi->Devmode == NULL)
	    goto cleanup;	    
	memcpy(pi->Devmode, &DefaultDevmode, sizeof(DefaultDevmode) );
	strcpy(pi->Devmode->dmPublic.dmDeviceName,name);
	DrvSetPrinterData16((LPSTR)name, (LPSTR)INT_PD_DEFAULT_DEVMODE,
		 REG_BINARY, (LPBYTE)&DefaultDevmode, sizeof(DefaultDevmode) );

	/* need to do something here AddPrinter?? */
    }
    else {
        pi->Devmode = HeapAlloc( PSDRV_Heap, 0, needed );
	DrvGetPrinterData16((LPSTR)name, (LPSTR)INT_PD_DEFAULT_DEVMODE, &type,
			  (LPBYTE)pi->Devmode, needed, &needed);
    }

    if (OpenPrinterA (pi->FriendlyName, &hPrinter, NULL) == 0) {
	ERR ("OpenPrinterA failed with code %li\n", GetLastError ());
	goto cleanup;
    }
    
    ppdFileName[0]='\0';
    
#ifdef HAVE_CUPS
    {
	ppd = cupsGetPPD(name);

	if (ppd) {
	    strncpy(ppdFileName, ppd, sizeof(ppdFileName));
	    res = ERROR_SUCCESS;
	    /* we should unlink() that file later */
	} else {
	    res = ERROR_FILE_NOT_FOUND;
	    WARN("Did not find ppd for %s\n",name);
	}
    }
#endif

    if (!ppdFileName[0]) {
        res = GetPrinterDataA (hPrinter, "PPD File", NULL, ppdFileName,
	    	sizeof(ppdFileName), &needed);
    }
    /* Look for a ppd file for this printer in the config file.
     * First look for the names of the printer, then for 'generic'
     */
    if((res != ERROR_SUCCESS) && !RegOpenKeyA(HKEY_LOCAL_MACHINE, "Software\\Wine\\Wine\\Config\\ppd", &hkey))
    {
	DWORD count = sizeof(ppdFileName);
	ppdFileName[0] = 0;
	if(RegQueryValueExA(hkey, name, 0, &type, ppdFileName, &count) != ERROR_SUCCESS)
	    RegQueryValueExA(hkey, "generic", 0, &type, ppdFileName, &count);
	RegCloseKey(hkey);
    }

    if(!ppdFileName[0])
	res = ERROR_FILE_NOT_FOUND;
    else 
	res = ERROR_SUCCESS;

    if (res != ERROR_SUCCESS) {
	ERR ("Error %li getting PPD file name for printer '%s'\n", res, name);
	goto closeprinter;
    }
    
    ppdFileName[sizeof(ppdFileName) - 1] = '\0';
    
    pi->ppd = PSDRV_ParsePPD(ppdFileName);
    if(!pi->ppd) {
	MESSAGE("Couldn't find PPD file '%s', expect a crash now!\n",
	    ppdFileName);
	goto closeprinter;
    }
    
    /*
     *	This is a hack.  The default paper size should be read in as part of
     *	the Devmode structure, but Wine doesn't currently provide a convenient
     *	way to configure printers.
     */
    res = GetPrinterDataA (hPrinter, "Paper Size", NULL, (LPBYTE) &dwPaperSize,
	    sizeof (DWORD), &needed);
    if (res == ERROR_SUCCESS)
	pi->Devmode->dmPublic.u1.s1.dmPaperSize = (SHORT) dwPaperSize;
    else if (res == ERROR_FILE_NOT_FOUND)
	TRACE ("No 'Paper Size' for printer '%s'\n", name);
    else {
	ERR ("GetPrinterDataA returned %li\n", res);
	goto closeprinter;
    }

    res = EnumPrinterDataExA (hPrinter, "PrinterDriverData\\FontSubTable", NULL,
	    0, &needed, &pi->FontSubTableSize);
    if (res == ERROR_SUCCESS || res == ERROR_FILE_NOT_FOUND) {
	TRACE ("No 'FontSubTable' for printer '%s'\n", name);
    }
    else if (res == ERROR_MORE_DATA) {
	pi->FontSubTable = HeapAlloc (PSDRV_Heap, 0, needed);
	if (pi->FontSubTable == NULL) {
	    ERR ("Failed to allocate %li bytes from heap\n", needed);
	    goto closeprinter;
	}

	res = EnumPrinterDataExA (hPrinter, "PrinterDriverData\\FontSubTable",
		(LPBYTE) pi->FontSubTable, needed, &needed,
		&pi->FontSubTableSize);
	if (res != ERROR_SUCCESS) {
	    ERR ("EnumPrinterDataExA returned %li\n", res);
	    goto closeprinter;
	}
    }
    else {
	ERR("EnumPrinterDataExA returned %li\n", res);
	goto closeprinter;
    }

    if (ClosePrinter (hPrinter) == 0) {
	ERR ("ClosePrinter failed with code %li\n", GetLastError ());
	goto cleanup;
    }

    pi->next = NULL;
    pi->Fonts = NULL;

    for(font = pi->ppd->InstalledFonts; font; font = font->next) {
        afm = PSDRV_FindAFMinList(PSDRV_AFMFontList, font->Name);
	if(!afm) {
	    TRACE( "Couldn't find AFM file for installed printer font '%s' - "
	    	    "ignoring\n", font->Name);
	}
	else {
	    BOOL added;
	    if (PSDRV_AddAFMtoList(&pi->Fonts, afm, &added) == FALSE) {
	    	PSDRV_FreeAFMList(pi->Fonts);
		goto cleanup;
	    }
	}

    }
    if (ppd) unlink(ppd);
    return pi;

closeprinter:
    ClosePrinter(hPrinter);
cleanup:
    if (pi->FontSubTable)
    	HeapFree(PSDRV_Heap, 0, pi->FontSubTable);
    if (pi->FriendlyName)
    	HeapFree(PSDRV_Heap, 0, pi->FriendlyName);
    if (pi->Devmode)
    	HeapFree(PSDRV_Heap, 0, pi->Devmode);
fail:
    HeapFree(PSDRV_Heap, 0, pi);
    if (ppd) unlink(ppd);
    *last = NULL;
    return NULL;
}
